/*
 * Copyright (c) 2008-2015 Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
/*global Ext,Sonatype,Nexus*/

/**
 * Repository UI form to enable/disable health check on individual repos.
 * @since 2.8
 */
NX.define('Nexus.healthcheck.view.HealthCheckPanel', {
    extend: 'Ext.Panel',
    mixins: [
      'Nexus.LogAwareMixin'
    ],
    title: 'Health Check',
    border: false,
    healtchCheckStatusStore: new Ext.data.SimpleStore({
      fields: ['value', 'display'],
      data: [
        ['enabled', 'true'],
        ['disabled', 'false']
      ]
    }),

    /**
     * @override
     */
    initComponent: function () {
      var me = this,
        onHealthCheckStatusUpdated = function(repositoryId, enabled) {
          if (me.repositoryId === repositoryId) {
            // reload settings to apply view customizations
            me.loadConfiguration();
          }
        };

      Ext.apply(me, {
        listeners: {
          activate: me.loadConfiguration,
          destroy: function() {
            Sonatype.Events.removeListener('healthCheckStatusUpdated', onHealthCheckStatusUpdated);
          }
        },
        items: [
          {
            xtype: 'form',
            border: false,
            style: {
              padding: '10px'
            },

            defaults: {
              hideLabel: true,
              inputValue: 'true' // use 'true' instead of 'on'
            },
            items: [
              {
                xtype: 'fieldset',
                checkboxToggle: false,
                title: 'Configure Repository Health Check',
                anchor: Sonatype.view.FIELDSET_OFFSET,
                collapsible: false,
                autoHeight: true,
                layoutConfig: {
                  labelSeparator: ''
                },
                items: [
                  {
                    xtype: 'combo',
                    lazyInit: false,
                    fieldLabel: 'Enabled',
                    itemCls: 'required-field',
                    helpText: 'Enable or disable Health Check for this repository',
                    name: 'status',
                    store: this.healtchCheckStatusStore,
                    valueField: 'value',
                    displayField: 'display',
                    editable: false,
                    forceSelection: true,
                    mode: 'local',
                    triggerAction: 'all',
                    selectOnFocus: true,
                    allowBlank: false,
                    minListWidth: 100
                  }
                ]
              }
            ],
            buttonAlign: 'left',
            buttons: [
              {
                xtype: 'button',
                text: 'Save',
                formBind: true,
                scope: me,
                handler: me.maybeSaveConfiguration
              },
              {
                xtype: 'button',
                text: 'Reset',
                formBind: true,
                scope: me,
                handler: me.loadConfiguration
              }
            ]
          }
        ]
      });
      me.constructor.superclass.initComponent.apply(me, arguments);

      Sonatype.Events.addListener('healthCheckStatusUpdated', onHealthCheckStatusUpdated);
    },

    /**
     * load existing configuration for a repository
     * @private
     */
    loadConfiguration: function() {
      var me = this;
      me.logDebug('loading configuration');
      me.loadEulaStatus();
      Ext.Ajax.request({
        url: Sonatype.HealthCheckUtil.getStatusUrl(null, me.repositoryId),
        method: 'GET',
        scope: me,
        success: function (response, opts) {
          me.logDebug('Configuration: ' + response.responseText);
          me.setValues(Ext.decode(response.responseText));
        }
      });
    },

    /**
     * persist the configuration changes
     * @param {object} data
     * @private
     */
    saveConfiguration: function(data) {
      var me = this;
      Ext.Ajax.request({
        url: Sonatype.HealthCheckUtil.getStatusUrl(me.rhcEulaAccepted),
        method: 'PUT',
        jsonData: Ext.apply(data, {id: me.repositoryId}),
        scope: me,
        success: function() {
          me.showMessage(data.status + ' for ' + me.repositoryId);

          Sonatype.Events.fireEvent('healthCheckStatusUpdated', me.repositoryId, 'enabled' === data.status);
        }
      });
    },

    /**
     * @private
     */
    maybeSaveConfiguration: function(button) {
      var me = this,
        values = me.getValues();
      values.status = values.status === 'true' ? 'enabled' : 'disabled';
      me.logDebug('Saving settings: ' + Ext.util.JSON.encode(values));

      if(!me.rhcEulaAccepted){
        me.showEula(values);
        return;
      }
      me.saveConfiguration(values);
    },

    /**
     * Check to see if we have already accepted the eula or not.
     * @private
     */
    loadEulaStatus: function(){
      var me = this;
      Ext.Ajax.request({
        url: Sonatype.HealthCheckUtil.getBase()+ '/healthCheckEula',
        method: 'GET',
        scope: me,
        success: function (response, opts) {
          me.logDebug('Eula acceptance: ' + response.responseText);
          me.rhcEulaAccepted = Ext.decode(response.responseText).accepted;
        }
      });
    },

    /**
     * Show the eula and save a configuration change only if it is accepted.
     * @param {object} data
     */
    showEula: function(data) {
      var me = this, accepted = false;
      var msg = '<span style="font-size: 12px">The analysis of the repository ' + me.repositoryId
        + ' for security vulnerabilities and license issues is regulated'
        + ' by the following terms of use. Please read carefully and press "I Agree" to continue.</span>';
      me.rhcEulaWindow = NX.create('Ext.Window',{
        title: 'CLM Terms of Use',
        closable: true,
        closeAction: 'hide',
        autoWidth: false,
        width: 725,
        autoHeight: true,
        autoScroll: false,
        modal: true,
        constrain: true,
        resizable: false,
        draggable: false,
        items: [
          {
            xtype: 'panel',
            frame: true,
            html: msg
          },
          {
            xtype: 'panel',
            autoScroll: true,
            height: 450,
            items: [
              {
                xtype: 'panel',
                autoLoad: {
                  url: Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('healthcheck-tos.html'),
                  headers: {
                    'accept': 'text/html'
                  }
                }
              },
              {
                xtype: 'panel',
                html: '<a href="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('healthcheck-tos.html') +
                  '" target="_new">Download copy of TOS</a>'
              }
            ]
          }
        ],
        buttonAlign: 'center',
        buttons: [
          {
            text: 'I Agree',
            handler: function() {
              me.rhcEulaAccepted = true;
              me.saveConfiguration(data);
              me.rhcEulaWindow.hide();
            }
          },
          {
            text: 'I Don\'t Agree',
            handler: function() {
              me.rhcEulaWindow.hide();
              me.showMessage('EULA refused, changes are not saved');
            }
          }
        ]
      });
      me.rhcEulaWindow.show();
    },

    /**
     * Helper to show Health Check message.
     *
     * @private
     */
    showMessage: function(message) {
      Nexus.messages.show('Health Check', message);
    },

    /**
     * Get form values.
     *
     * @public
     */
    getValues: function () {
      return this.down('form').getForm().getValues();
    },

    /**
     * Set form values.
     *
     * @public
     */
    setValues: function(values) {
      this.down('form').getForm().setValues(values);
    },

    /**
     * finds a field by name in the form
     * @private
     * @param {string} fieldName
     */
    field : function(fieldName){
      return this.down('form').getForm().findField(fieldName);
    }
  }
);