/*
 * Copyright (c) 2008-2015 Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
define('Nexus/healthcheck/HealthCheckSearchContrib', [ 'Nexus/healthcheck/HealthCheckUtil' ], function() {

  HealthCheckSearchContrib = function(config) {
      var me = this;
      me.reset = function(config) {
          var config = config || {};
          var defaultConfig = {
              extraDataQueryComplete : false,
              rhcSearchResultMap : [],
              rhcResultDataCapped : false,
              rhcResultDataError : false,
              rhcResultDataDisabled : false,
              rhcAlertColumns : Sonatype.HealthCheckProUtil && Sonatype.HealthCheckProUtil.hasNonFreeProLicense(),
              rhcAlertData : false,
              grid : null
          };
          Ext.apply(me, config, defaultConfig);

          if (me.grid != null) {
              me.updateStore();
              me.updateColumnModel();
          }
      };

      me.reset(config);

      me.updateStore = function() {
          var grid = me.grid;
          var store = grid.store;

          me.addFieldToStore('age');
          me.addFieldToStore('popularity');
          me.addFieldToStore('mostPopularVersion');
          if (me.rhcAlertColumns) {
              me.addFieldToStore('securityAlerts');
              me.addFieldToStore('licenseThreat');
          }

          // store.fields is readonly in ExtJS3, so we need to explicitely rebuild the field extractors
          delete store.reader.ef;
          store.reader.buildExtractors();

          store.addListener('beforeload', me.storeBeforeLoadListener);
          store.addListener('load', me.storeLoadListener);
      };

      me.updateColumnModel = function() {
          //this is solely used because you can't apply scope to the column renderer functions
          me.grid.columns.splice(3, 0, {
              header : 'Age',
              id : 'artifactAgeCountColumn',
              dataIndex : 'age',
              sortable : true,
              hidden : true,
              renderer : function(value, metadata, record, rowIndex, colIndex, store) {
                  return me.buildAgeText(record);
              }
          }, {
              header : 'Popularity',
              id : 'artifactPopularityCountColumn',
              dataIndex : 'popularity',
              sortable : true,
              hidden : true,
              renderer : function(value, metadata, record, rowIndex, colIndex, store) {
                  metadata.css = 'rhcMiddleAlignCell';
                  return me.buildPopularityBar(record);
              }
          }, {
              header : 'Most Popular Version',
              id : 'artifactMostPopularVersionColumn',
              dataIndex : 'mostPopularVersion',
              sortable : true,
              hidden : true,
              renderer : function(value, metadata, record, rowIndex, colIndex, store) {
                  return me.buildMostPopularVersion(record);
              }
          });
          if (me.rhcAlertColumns) {
              me.grid.columns.splice(6, 0, {
                  header : 'Security Issues',
                  id : 'artifactSecurityAlertsColumn',
                  dataIndex : 'securityAlerts',
                  sortable : true,
                  hidden : true,
                  renderer : function(value, metadata, record, rowIndex, colIndex, store) {
                      metadata.css = 'rhcMiddleAlignCell rhcSecurityAlertsCell';
                      return me.buildSecurityAlerts(record);
                  }
              }, {
                  header : 'License Threat',
                  id : 'artifactLicenseThreatColumn',
                  dataIndex : 'licenseThreat',
                  sortable : true,
                  hidden : true,
                  renderer : function(value, metadata, record, rowIndex, colIndex, store) {
                      metadata.css = 'rhcMiddleAlignCell rhcLicenseThreatCell';
                      return me.buildLicenseThreat(record);
                  }
              });
          }
      };

      me.addFieldToStore = function(name) {
          me.grid.store.fields.add(new Ext.data.Field({
              name : name
          }));
      };

      me.storeBeforeLoadListener = function() {
          me.extraDataQueryComplete = false;
          me.rhcAlertData = me.rhcAlertColumns && Sonatype.HealthCheckProUtil.hasPaidProLicense() && Sonatype.HealthCheckProUtil.hasAlertUpdatePerm();
      };

      me.storeLoadListener = function(store, records, options) {
          me.rhcResultDataCapped = false;
          me.rhcResultDataError = false;
          me.rhcResultDataDisabled = false;
          me.rhcSearchResultMap = [];

          if (store.reader.jsonData.collapsed) {
              me.handleGASearchResults(records);
          } else {
              me.handleGAVSearchResults(records);
          }
      };

      me.getGavData = function(record) {
          var data = me.rhcSearchResultMap[record.get('groupId') + '|||' + record.get('artifactId') + '|||' + record.get('version')];
          return data;
      };

      me.getGaData = function(record) {
          var data = me.rhcSearchResultMap[record.get('groupId') + '|||' + record.get('artifactId')];
          return data;
      };

      me.notAvailable = function() {
          return '<img class="opaqueWarning" src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/na.png') + '">';
      };

      me.buildAgeText = function(record) {
          var data = me.getGavData(record);

          var result = me.buildCommonText(data, false);
          if (result != null) {
              return result;
          } else if (data && (data.age === 0 || data.age > 0)) {
              // convert millis to a day count
              var dayAge = data.age / (1000 * 60 * 60 * 24);

              if (dayAge > 364) {
                  return (dayAge / 365).toFixed(1) + ' yrs';
              } else {
                  return dayAge.toFixed(0) + ' d';
              }
          } else {
              return me.notAvailable();
          }
      };

      me.buildPopularityBar = function(record) {
          var data = me.getGavData(record);

          var result = me.buildCommonText(data, false);
          if (result != null) {
              return result;
          } else if (data && (data.popularity === 0 || data.popularity > 0)) {
              if (data.popularity > 100) {
                  data.popularity = 100;
              }

              return '<p style="white-space: nowrap; padding-right: 2px;"><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/bar_start.gif')
                      + '" style="vertical-align: middle;"/><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/blue_body.gif') + '" width="' + data.popularity
                      + '%" height="8px" style="vertical-align: middle;"/><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/blue_border.gif')
                      + '" style="vertical-align: middle;"/></p>';
          } else {
              return me.notAvailable();
          }
      };

      me.buildMostPopularVersion = function(record) {
          var data = me.getGaData(record);

          var result = me.buildCommonText(data, false);
          if (result != null) {
              return result;
          } else if (data && !Ext.isEmpty(data.mostPopularVersion)) {
              return data.mostPopularVersion;
          } else {
              return me.notAvailable();
          }
      };

      function buildChiclet(clazz, value) {
        return '<span class="nx-clm-chiclet-small ' + clazz + '"><strong>' + value + '</strong></span>';
      }

      me.buildSecurityAlerts = function(record) {
          var data = me.getGavData(record);

          var result = me.buildCommonText(data, true);
          if (result != null) {
              return result;
          } else if (data &&
              !(Ext.isEmpty(data.criticalSecurityAlerts) && Ext.isEmpty(data.severeSecurityAlerts)
                  && Ext.isEmpty(data.moderateSecurityAlerts) && Ext.isEmpty(data.noneSecurityAlerts))) {
              var clazz = data.criticalSecurityAlerts > 0 ? 'critical' : '';
              var value = data.criticalSecurityAlerts > 0 ? data.criticalSecurityAlerts : '';
              var html = buildChiclet(clazz, value);
              clazz = data.severeSecurityAlerts > 0 ? 'severe' : '';
              value = data.severeSecurityAlerts > 0 ? data.severeSecurityAlerts : '';
              html += buildChiclet(clazz, value);
              clazz = data.moderateSecurityAlerts > 0 ? 'moderate' : '';
              value = data.moderateSecurityAlerts > 0 ? data.moderateSecurityAlerts : '';
              html += buildChiclet(clazz, value);
              return html;
          } else {
              return me.notAvailable();
          }
      };

      me.buildLicenseThreat = function(record) {
          var data = me.getGavData(record);

          var result = me.buildCommonText(data, true);
          if (result != null) {
              return result;
          } else if (data && !(Ext.isEmpty(data.licenseThreat) && Ext.isEmpty(data.licenseThreatName))) {
              var barClass = '';
              if (data.licenseThreat == 4) {
                  barClass = 'critical';
              } else if (data.licenseThreat == 2) {
                  barClass = 'moderate';
              } else if (data.licenseThreat == 1) {
                  barClass = 'none';
              } else {
                  barClass = 'severe';
              }
              return '<span class="nx-clm-bar ' + barClass + '"><strong>' + data.licenseThreatName + '</strong></span>';
          } else {
              return me.notAvailable();
          }
      };

      me.buildCommonText = function(data, alerts) {
          if (alerts && !me.rhcAlertData) {
              return '<img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/locked.png') + '">';
          } else if (!me.extraDataQueryComplete) {
              return 'Loading...';
          } else if (me.rhcResultDataDisabled) {
              return '<img class="opaqueWarning" src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/information.png') + '">';
          } else if (me.rhcResultDataError) {
              return '<img class="opaqueWarning" src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/exclamation.gif') + '">';
          } else if (me.rhcResultDataCapped || (data && data[alerts ? 'cappedAlerts' : 'capped'])) {
              return '<img class="opaqueWarning" src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/warning.gif') + '">';
          } else {
              return null;
          }
      };

      me.handleGASearchResults = function(records) {
          me.grid.getColumnModel().setHidden(3, true);
          me.grid.getColumnModel().setHidden(4, true);
          me.grid.getColumnModel().setHidden(5, false);
          if (me.rhcAlertColumns) {
              me.grid.getColumnModel().setHidden(6, true);
              me.grid.getColumnModel().setHidden(7, true);
          }

          var data = {
              gas : []
          };

          if (records) {
              for ( var i = 0; i < records.length; i++) {
                  data.gas.push({
                      groupId : records[i].get('groupId'),
                      artifactId : records[i].get('artifactId')
                  });
              }
          }

          Ext.Ajax.request({
              callback : me.gaDetailsCallback,
              suppressStatus : [ 500, 412, 409 ],
              jsonData : data,
              method : 'PUT',
              url : Sonatype.HealthCheckUtil.getBase() + '/gaDetails'
          });
      };

      me.handleGAVSearchResults = function(records) {
          me.grid.getColumnModel().setHidden(3, false);
          me.grid.getColumnModel().setHidden(4, false);
          me.grid.getColumnModel().setHidden(5, true);
          if (me.rhcAlertColumns) {
              me.grid.getColumnModel().setHidden(6, false);
              me.grid.getColumnModel().setHidden(7, false);
          }

          var data = {
              gavs : []
          };

          if (records) {
              for ( var i = 0; i < records.length; i++) {
                  data.gavs.push({
                      groupId : records[i].get('groupId'),
                      artifactId : records[i].get('artifactId'),
                      version : records[i].get('version')
                  });
              }
          }

          Ext.Ajax.request({
              callback : me.gavDetailsCallback,
              suppressStatus : [ 500, 412, 409 ],
              jsonData : data,
              method : 'PUT',
              url : Sonatype.HealthCheckUtil.getBase() + (me.rhcAlertData ? '/gavDetailsPro' : '/gavDetails')
          });
      };

      me.gaDetailsCallback = function(options, success, response) {
          me.extraDataQueryComplete = true;
          me.updateTooltip();
          if (success) {
              var json = Ext.decode(response.responseText);

              if (json && json.gas) {
                  for ( var i = 0; i < json.gas.length; i++) {
                      me.rhcSearchResultMap[json.gas[i].groupId + '|||' + json.gas[i].artifactId] = json.gas[i];
                  }
              }
          } else if (response.status == 409) {
              me.rhcResultDataCapped = true;
          } else if (response.status == 412) {
              me.rhcResultDataDisabled = true;
          } else if (response.status == 500) {
              me.rhcResultDataError = true;
          }

          me.grid.store.each(function(rec, index) {
              var data = me.getGaData(rec);
              if (data && data.mostPopularVersion) {
                  //I don't want to send events on this commit, waste all kinds of time, simply want the data in the record updated
                  me.grid.store.suspendEvents();
                  rec.set('mostPopularVersion', data.mostPopularVersion);
                  rec.commit();
                  me.grid.store.resumeEvents();
              }
              var val = me.buildMostPopularVersion(rec);
              Ext.fly(me.grid.getView().getCell(index, 5).firstChild).update(val);
          });
      };

      me.gavDetailsCallback = function(options, success, response) {
          me.extraDataQueryComplete = true;
          me.updateTooltip();
          if (success) {
              var json = Ext.decode(response.responseText);

              if (json && json.gavs) {
                  for ( var i = 0; i < json.gavs.length; i++) {
                      me.rhcSearchResultMap[json.gavs[i].groupId + '|||' + json.gavs[i].artifactId + '|||' + json.gavs[i].version] = json.gavs[i];
                  }
              }
          } else if (response.status == 409) {
              me.rhcResultDataCapped = true;
          } else if (response.status == 412) {
              me.rhcResultDataDisabled = true;
          } else if (response.status == 500) {
              me.rhcResultDataError = true;
          }

          me.grid.store.each(function(rec, index) {
              var data = me.getGavData(rec);
              if (data && (data.age || data.popularity || data.securityAlerts || data.licenseThreat)) {
                  //I don't want to send events on this commit, waste all kinds of time, simply want the data in the record updated
                  me.grid.store.suspendEvents();
                  rec.set('age', data.age);
                  rec.set('popularity', data.popularity);
                  if (me.rhcAlertColumns) {
                      rec.set('securityAlerts', data.securityAlerts);
                      rec.set('licenseThreat', data.licenseThreat);
                  }
                  rec.commit();
                  me.grid.store.resumeEvents();
              }
              var val = me.buildAgeText(rec);
              Ext.fly(me.grid.getView().getCell(index, 3).firstChild).update(val);
              val = me.buildPopularityBar(rec);
              Ext.fly(me.grid.getView().getCell(index, 4).firstChild).update(val);
              if (me.rhcAlertColumns) {
                  val = me.buildSecurityAlerts(rec);
                  Ext.fly(me.grid.getView().getCell(index, 6).firstChild).update(val);
                  val = me.buildLicenseThreat(rec);
                  Ext.fly(me.grid.getView().getCell(index, 7).firstChild).update(val);
              }
          });
      };

      me.updateTooltip = function() {
          var store = me.grid.getStore();
          var view = me.grid.getView();
          if (me.grid.tip) {
              me.grid.tip.destroy();
          }
          me.grid.tip = new Ext.ToolTip({
              target : view.mainBody,
              delegate : '.x-grid3-cell',
              maxWidth : 500,
              mouseOffset : [ 0, 0 ],
              renderTo : document.body,
              listeners : {
                  beforeshow : {
                      fn : me.tooltipBeforeShowListener
                  }
              }
          });
      };

      me.tooltipBeforeShowListener = function(tip) {
          tip.hideDelay = 200;
          var view = me.grid.getView();
          var rowIndex = view.findRowIndex(tip.triggerElement);
          var cellIndex = view.findCellIndex(tip.triggerElement);

          var record = me.grid.store.getAt(rowIndex);

          if (!record) {
              return false;
          }

          var tooltipText = null;

          var colModel = me.grid.getColumnModel();

          var rhcCol = cellIndex == colModel.getIndexById('artifactAgeCountColumn') || cellIndex == colModel.getIndexById('artifactPopularityCountColumn')
                  || cellIndex == colModel.getIndexById('artifactMostPopularVersionColumn');
          var rhcColAlert = !rhcCol && me.rhcAlertColumns && (cellIndex == colModel.getIndexById('artifactSecurityAlertsColumn')
                  || cellIndex == colModel.getIndexById('artifactLicenseThreatColumn'));

          if (rhcCol) {
              var data = me.getGavData(record);
              if (!data) {
                  data = me.getGaData(record);
              }

              if (me.rhcResultDataDisabled) {
                  tooltipText = 'The age and popularity data is only available once Repository Health Check (RHC) has been enabled.';
              } else if (me.rhcResultDataError) {
                  tooltipText = 'Error retrieving data from Sonatype CLM';
              } else if (me.rhcResultDataCapped || (data && data.capped)) {
                  tooltipText = 'The query limit for age and popularity data has been reached.  '
                          + 'Contact <a href="http://sonatype.com/support" target="_blank">Sonatype support</a> to extend current quota limits.';
                  tip.hideDelay = 3000;
              } else if (!data || Ext.isEmpty(data[colModel.getDataIndex(cellIndex)])) {
                  tooltipText = 'CLM data is not available for this artifact. Note that internal artifacts are not covered in Sonatype CLM.';
              }
          } else if (rhcColAlert) {
              var data = me.getGavData(record);

              if (!me.rhcAlertData) {
                  if (!Sonatype.HealthCheckProUtil.hasPaidProLicense()) {
                      tooltipText = 'License and security data is available to Nexus Professional customers.';
                  } else if (Sonatype.user.curr.isLoggedIn) {
                      tooltipText = 'You do not have permission to view the license and security data.';
                  } else {
                      tooltipText = 'The license and security data is only available once you have logged in.';
                  }
              } else if (me.rhcResultDataDisabled) {
                  tooltipText = 'The license and security data is only available once Repository Health Check (RHC) has been enabled.';
              } else if (me.rhcResultDataError) {
                  tooltipText = 'Error retrieving data from Sonatype CLM';
              } else if (me.rhcResultDataCapped || (data && data.cappedAlerts)) {
                  tooltipText = 'The query limit for license and security data has been reached.  '
                          + 'Contact <a href="http://sonatype.com/support" target="_blank">Sonatype support</a> to extend current quota limits.';
                  tip.hideDelay = 3000;
              } else if (!data || Ext.isEmpty(data[colModel.getDataIndex(cellIndex)])) {
                  tooltipText = 'CLM data is not available for this artifact. Note that internal artifacts are not covered in Sonatype CLM.';
              }
          }

          if (Ext.isEmpty(tooltipText)) {
              tooltipText = record.get('highlightedFragment');
          }

          if (Ext.isEmpty(tooltipText)) {
              return false;
          }

          tip.body.dom.innerHTML = tooltipText;
      };

      Sonatype.Events.addListener('searchResultGridInit', function(grid) {
          me.reset({
              grid : grid
          });
      });
  };

  Ext.onReady(function() {
      Sonatype.HealthCheckSearchContrib = new HealthCheckSearchContrib();
  });
});
