/*
 * Copyright (c) 2008-2015 Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
define('Nexus/healthcheck/HealthCheckRepositoryContrib', [ 'Nexus/healthcheck/HealthCheckUtil' ], function() {

  HealthCheckRepositoryContrib = function(config) {
      var me = this;
      me.reset = function(config) {
          var config = config || {};
          var defaultConfig = {
              isHealthCheckUpdate : false,
              healthCheckWindow : null,
              healthCheckStatusTask : null,
              healthCheckDataMap : [],
              healthCheckCellMouseRecordId : false,
              rhcAnalyzingState : true,
              rhcEulaAccepted : false,
              rhcConfiguredForNew : false,
              rhcToolbarHeight : 26,
              healthCheckStatusError : false,
              healthCheckTooltip : null,
              grid : null
          };
          Ext.apply(me, config, defaultConfig);

          if (me.grid) {
              me.grid.store.addListener('load', function(store, records, options) {
                  // force at least 1 request
                  me.rhcAnalyzingState = true;
                  me.startRHCPollTask();
                  me.generateHealthCheckTooltip();
              });

              me.grid.store.addListener('add', function(store, records, index) {
                  if (records.length == 1) {
                      if (!me.healthCheckDataMap[records[0].get('id')] && me.isProxyRepository(records[0])) {
                          var healthCheckData = {};
                          healthCheckData.status = me.rhcConfiguredForNew ? 'analyzing' : 'disabled';
                          me.healthCheckDataMap[records[0].get('id')] = healthCheckData;
                      }
                  }
              });

              me.grid.mon(me.grid, 'mouseover', function(e, t) {
                  var row = me.grid.getView().findRowIndex(t);
                  var col = me.grid.getView().findCellIndex(t);
                  me.healthCheckCellMouseRecordId = null;
                  if (row !== false && col == me.grid.getColumnModel().getIndexById('healthCheckColumn')) {
                      var record = me.grid.store.getAt(row);
                      var id = record.get('id');
                      var healthCheckData = me.healthCheckDataMap[id];
                      if (!me.healthCheckStatusError && healthCheckData && healthCheckData.status == 'enabled' && me.isRepositoryInService(record)) {
                          me.healthCheckCellMouseRecordId = id;
                          me.openSummary(Ext.get(me.grid.getView().getCell(row, col)), record);
                      }
                  }
              });

              me.grid.mon(me.grid, 'mouseout', function(e, t) {
                  me.healthCheckCellMouseRecordId = null;
              });
          }
      };

      me.reset(config);

      me.isRepositoryInService = function(record) {
          var status = record.get('status');
          if (status) {
              status = status.localStatus;
          }
          return status == 'IN_SERVICE';
      };

      me.isProxyRepository = function(record) {
          return record.get('repoType') == 'proxy';
      };

      me.isUnsupportedRepository = function(record) {
        return !me.isProxyRepository(record) || record.get('repoPolicy') == 'Snapshot' ||  !me.isRepositoryInService(record);
      };

      me.showEula = function(acceptHandler) {
          var msg = '<span style="font-size: 12px">The analysis of repositories'
                  + ' for security vulnerabilities and license issues is regulated'
                  + ' by the following terms of use. Please read carefully and press "I Agree" to continue.</span>',
              rhcEulaWindow = new Ext.Window({
              title : 'CLM Terms of Use',
              closable : true,
              closeAction : 'hide',
              autoWidth : false,
              width : 725,
              autoHeight : true,
              autoScroll : false,
              modal : true,
              constrain : true,
              resizable : false,
              draggable : false,
              items : [ {
                  xtype : 'panel',
                  frame : true,
                  html : msg
              }, {
                  xtype : 'panel',
                  autoScroll : true,
                  height : 450,
                  items : [ {
                      xtype : 'panel',
                      autoLoad : {
                          url : Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('healthcheck-tos.html'),
                          headers : {
                              'accept' : 'text/html'
                          }
                      }
                  }, {
                      xtype : 'panel',
                      html : '<a href="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('healthcheck-tos.html') + '" target="_new">Download copy of TOS</a>'
                  } ]
              } ],
              buttonAlign : 'center',
              buttons : [ {
                  text : 'I Agree',
                  handler : function() {
                      rhcEulaWindow.hide();
                      acceptHandler();
                  }
              }, {
                  text : 'I Don\'t Agree',
                  handler : function() {
                      rhcEulaWindow.hide();
                  }
              } ]
          });
          rhcEulaWindow.show();
      };

      me.updateColumns = function(columns) {
          columns.splice(8, 0, {
              name : 'resourceURI',
              header : 'Health Check',
              id : 'healthCheckColumn',
              width : 100,
              renderer : function(value, metadata, record, rowIndex, colIndex, store) {
                  var healthCheckData = me.healthCheckDataMap[record.get('id')];

                  if (me.healthCheckStatusError) {
                      return '<div><img class="opaqueWarning" src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/exclamation.png') + '"></div>';
                  } else if (me.isUnsupportedRepository(record)) {
                      return '<div><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/analyze_disabled.png') + '"></div>';
                  } else if (!healthCheckData) {
                      return 'Loading...';
                  } else if (healthCheckData.status == 'enabled') {
                      // enabled, so add link to open iframe
                      metadata.css = 'cursorPointerCell badgeCell';
                      return '<div><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/security-alert.png') + '">&nbsp;'
                              + healthCheckData.securityIssueCount + '&nbsp;&nbsp;<img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/license-alert.png')
                              + '" style="margin-left:10px">&nbsp;' + healthCheckData.licenseIssueCount + '</div>';
                  } else if (Sonatype.HealthCheckUtil.hasEditPerm() && healthCheckData.status == 'disabled') {
                      // update privilege and disabled, so allow to enable
                      metadata.css = 'cursorPointerCell';
                      return '<div><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/analyze.png') + '"></div>';
                  } else if (healthCheckData.status == 'disabled') {
                      return '<div><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/analyze_disabled.png') + '"></div>';
                  } else if (healthCheckData.status == 'analyzing') {
                      return '<div><img src="' + Sonatype.HealthCheckUtil.getHealthCheckResourceUrl('images/analyzing.png') + '"></div>';
                  }
              },
              clickHandler : function(grid, rowIndex, eventObject) {
                  var record = me.grid.getStore().getAt(rowIndex);
                  var healthCheckData = me.healthCheckDataMap[record.get('id')];

                  if (healthCheckData) {
                      // note that this wont really be used, as main way in is the hover listener, but left here for
                      // non-hover interfaces (i.e. ipad)
                      if (me.isUnsupportedRepository(record)) {
                          // eat it
                      } else if (!me.healthCheckStatusError && healthCheckData.status == 'enabled') {
                          var el = Ext.get(me.grid.getView().getCell(rowIndex, me.grid.getColumnModel().getIndexById('healthCheckColumn')));
                          me.healthCheckSummaryClick(record, el.getLeft(), el.getTop());
                      } else if (healthCheckData.status == 'disabled' && Sonatype.HealthCheckUtil.hasEditPerm()) {
                          Sonatype.MessageBox.show({
                              title : 'Analyze Repository',
                              msg : 'Do you want to analyze the repository ' + Ext.util.Format.htmlEncode(record.get('name'))
                                      + ' and others for security vulnerabilities and license issues?',
                              buttons : { ok : 'Yes, all repositories', yes : 'Yes, only this repository', no : true },
                              icon : Sonatype.MessageBox.QUESTION,
                              fn : function(btnId) {
                                  var recordToEnable = btnId === 'yes' ? record : (btnId === 'ok' ? 'all' : null);
                                  if (recordToEnable) {
                                      if (!me.rhcEulaAccepted) {
                                          me.showEula(function() {
                                              me.updateHealthCheck(recordToEnable, true, true);
                                          });
                                      } else {
                                          me.updateHealthCheck(recordToEnable, true);
                                      }
                                  }
                              }
                          });
                          if (me.healthCheckTooltip) {
                              me.healthCheckTooltip.hide();
                          }
                      }
                  }
              }
          });
      };

      me.openSummary = function(cellEl, record, deferred) {
          if (deferred) {
              if (me.healthCheckCellMouseRecordId === record.get('id')) {
                  me.healthCheckSummaryClick(record, cellEl.getLeft(), cellEl.getTop());
              }
          } else {
              me.openSummary.defer(500, me, [ cellEl, record, true ]);
          }
      };

      me.startRHCPollTask = function() {
          me.healthCheckStatusTask = {
              run : function() {
                  // if another task has started or if panel isn't around, return false to stop the task
                  if (!Ext.get(this.gridComponentId) || me.healthCheckStatusTask != this) {
                      return false;
                  }
                  // if nothing is 'analyzing' simply return and check next time (note this will not stop the task, only returning false will)
                  if (!me.rhcAnalyzingState) {
                      return;
                  }
                  Ext.Ajax.request({
                      callback : function(options, success, response) {
                          if (success) {
                              me.healthCheckStatusError = false;
                              var json = Ext.decode(response.responseText);
                              var store = me.grid.getStore();
                              me.rhcAnalyzingState = false;
                              me.rhcEulaAccepted = json.eulaAccepted;
                              me.rhcConfiguredForNew = json.configuredForNewRepositories;
                              for ( var i = 0; i < json.repositories.length; i++) {
                                  if (json.repositories[i].status == 'analyzing') {
                                      me.rhcAnalyzingState = true;
                                  }
                                  var record = me.findRecord(store, json.repositories[i].id);
                                  if (record) {
                                      var data = me.healthCheckDataMap[record.get('id')];
                                      me.healthCheckDataMap[record.get('id')] = json.repositories[i];

                                      // TODO: really need a good js deep equals here...
                                      if (!data || data.healthCheckSummaryUrl != json.repositories[i].healthCheckSummaryUrl || data.iframeWidth != json.repositories[i].iframeWidth
                                              || data.iframeHeight != json.repositories[i].iframeHeight || data.status != json.repositories[i].status
                                              || data.securityIssueCount != json.repositories[i].securityIssueCount
                                              || data.licenseIssueCount != json.repositories[i].licenseIssueCount) {
                                          me.grid.getView().refreshRow(record);
                                      }
                                  }
                              }
                          } else {
                              if (!me.healthCheckStatusError) {
                                  me.healthCheckStatusError = true;
                                  me.grid.getView().refresh();
                              }
                              //don't want to continue to poll while in error state
                              me.rhcAnalyzingState = false;
                          }
                      },
                      url : Sonatype.HealthCheckUtil.getStatusUrl(),
                      // hide the status when connection to host is lost
                      suppressStatus : [ 0, 500 ]
                  });
              },
              interval : 10000,
              gridComponentId : me.grid.getId()
          };

          Ext.TaskMgr.start(me.healthCheckStatusTask);
      };

      me.findRecord = function(store, repositoryId) {
          return store.getAt(store.findExact('id', repositoryId));
      };

      me.getHealthCheckSummarySizeParams = function(healthcheckdata) {
          // we are just assigning defaults here in case the cookie isn't available
          var sizeDeltas = Ext.state.Manager.get('healthcheck-summary-size-delta', {
              width : 0,
              height : 0
          });
          return {
              width : healthcheckdata.iframeWidth + sizeDeltas.width,
              height : healthcheckdata.iframeHeight + sizeDeltas.height
          };
      };

      me.healthCheckSummaryClick = function(record, x, y) {
          if (Sonatype.HealthCheckUtil.hasSummaryReadPerm()) {
              var bbarItems = [];

              var healthcheckdata = me.healthCheckDataMap[record.get('id')];

              var url = healthcheckdata.healthCheckSummaryUrl;

              if (!me.healthCheckWindow) {
                  Sonatype.Events.fireEvent('healthCheckWindowInit', me, bbarItems, record, healthcheckdata);

                  var sizeParams = me.getHealthCheckSummarySizeParams(healthcheckdata);

                  me.healthCheckWindow = new Ext.Window({
                      id : 'healthCheckWindow',
                      closable : false,
                      frame : false,
                      border : true,
                      layout : 'fit',
                      width : sizeParams.width,
                      height : sizeParams.height + (bbarItems.length > 0 ? me.rhcToolbarHeight : 0),
                      modal : false,
                      constrain : true,
                      resizable : true,
                      draggable : true,
                      x : x ? x : undefined,
                      y : y ? y : undefined,
                      bbar : bbarItems.length > 0 ? bbarItems : undefined,
                      items : [ {
                          xtype : 'iframepanel',
                          width : sizeParams.width,
                          height : sizeParams.height,
                          closable : true,
                          loadMask : {
                              msg : 'Loading Health Check Summary...'
                          },
                          autoShow : true,
                          id : 'healthCheckIframe',
                          defaultSrc : url,
                          disableMessaging : false
                      } ],
                      listeners : {
                          resize : {
                              fn : function(window, width, height) {
                                  Ext.state.Manager.set('healthcheck-summary-size-delta', {
                                      width : width - window.healthcheckdata.iframeWidth,
                                      height : height - (window.getBottomToolbar() ? me.rhcToolbarHeight : 0) - window.healthcheckdata.iframeHeight
                                  });
                              }
                          }
                      },
                      healthcheckdata : healthcheckdata
                  });
                  me.healthCheckWindow.show();
                  me.healthCheckWindow.el.on('mouseover', function() {
                      me.healthCheckWindow.windowMouse = true;
                  });
                  me.healthCheckWindow.el.on('mouseout', function() {
                      var resizeElement = Ext.getDom('healthCheckWindow-rzproxy');
                      if (Ext.isEmpty(resizeElement) || resizeElement.style["display"] != 'block') {
                          me.healthCheckWindow.windowMouse = false;
                      }
                  });

                  var closeFunction = function() {
                      if (me.healthCheckWindow) {
                          if ((!me.healthCheckWindow.windowMouse && !me.healthCheckCellMouseRecordId)) {
                              me.closeHealthCheckWindow();
                          } else {
                              this.defer(1000, this);
                          }
                      }
                  };
                  closeFunction.defer(1000, closeFunction);
              } else {
                  var iframeContainer = me.healthCheckWindow.getComponent('healthCheckIframe');
                  if (iframeContainer.defaultSrc != url) {
                      Sonatype.Events.fireEvent('healthCheckWindowInit', me, bbarItems, record, healthcheckdata);

                      me.healthCheckWindow.healthcheckdata = healthcheckdata;
                      iframeContainer.defaultSrc = url;
                      iframeContainer.setSrc();

                      var sizeParams = me.getHealthCheckSummarySizeParams(healthcheckdata);
                      me.healthCheckWindow.setSize(sizeParams.width, sizeParams.height + (bbarItems.length > 0 ? me.rhcToolbarHeight : 0));
                      if (x && y) {
                          me.healthCheckWindow.setPosition(x, y);
                      }

                      var bbar = me.healthCheckWindow.getBottomToolbar();
                      if (bbar) {
                          bbar.removeAll();
                          if (bbarItems.length > 0) {
                              for ( var i = 0; i < bbarItems.length; i++) {
                                  bbar.add(bbarItems[i]);
                              }
                          }
                      }
                  }
              }
          }
      };

      me.closeHealthCheckWindow = function() {
          if (me.healthCheckWindow != null) {
              me.healthCheckWindow.close();
              me.healthCheckWindow = null;
          }
      };

      me.updateHealthCheck = function(record, enabled, appendEula) {
          var opts = {};
          opts.url = Sonatype.HealthCheckUtil.getStatusUrl(appendEula);
          opts.callback = function(options, success, response) {
              if (enabled && success) {
                  // i only set this to true on success, just in case anything failed serverside, the user can do it again
                  me.rhcEulaAccepted = true;
              }
          };
          if ('all' === record) {
              opts.method = 'POST';
              opts.success = function(response, options) {
                var store = me.grid.store;
                for ( var i = 0; i < store.getCount(); i++ ) {
                    record = store.getAt(i);
                    if (me.healthCheckDataMap[record.get('id')]) {
                        Sonatype.Events.fireEvent('healthCheckStatusUpdated', record.get('id'), enabled, record);
                    }
                }
                me.rhcConfiguredForNew = true;
              };
          } else {
              opts.method = 'PUT';
              opts.jsonData = {
                  id : record.get('id'),
                  status : enabled ? 'enabled' : 'disabled'
              };
              opts.success = function(response, options) {
                  // refresh the column data
                  Sonatype.Events.fireEvent('healthCheckStatusUpdated', record.get('id'), enabled, record);
              };
          }
          Ext.Ajax.request(opts);
      };

      me.updateGridColumn = function(record, enabled) {
          var healthCheckData = me.healthCheckDataMap[record.get('id')];
          if (healthCheckData) {
              if (!enabled) {
                  healthCheckData.status = 'disabled';
                  if (me.healthCheckWindow != null && me.healthCheckWindow.healthcheckdata.id == healthCheckData.id) {
                      me.closeHealthCheckWindow();
                  }
              } else if (healthCheckData.status === 'disabled') {
                  healthCheckData.status = 'analyzing';
                  me.rhcAnalyzingState = true;
              }

              me.grid.getView().refreshRow(record);
          }
      };

      me.generateHealthCheckTooltip = function() {
          if (me.healthCheckTooltip) {
              me.healthCheckTooltip.destroy();
          }

          me.healthCheckTooltip = new Ext.ToolTip({
              target : me.grid.getView().mainBody,
              delegate : '.x-grid3-cell',
              maxWidth : 500,
              mouseOffset : [ 0, 0 ],
              renderTo : document.body,
              hideDelay : 5000,
              baseCls : 'x-tipcustom',
              cls : 'healthCheckTooltip',
              items : [ {
                  xtype : 'panel',
                  frame : true,
                  html : ''
              } ],
            listeners: {
              beforeshow: function(tip) {
                var rowIndex = me.grid.getView().findRowIndex(tip.triggerElement);
                var cellIndex = me.grid.getView().findCellIndex(tip.triggerElement);
                var record = me.grid.getStore().getAt(rowIndex);

                if (!record || Sonatype.MessageBox.isVisible()) {
                  return false;
                }

                if (cellIndex == me.grid.getColumnModel().getIndexById('healthCheckColumn')) {
                  var healthCheckData = me.healthCheckDataMap[record.get('id')];

                  if (me.healthCheckStatusError) {
                    tip.items.get(0).body.update('<span><h2>Repository Health Check Status Error</h2>' +
                      'Nexus was unable to successfully retrieve the Repository Health Check status of the repositories.');
                    return;
                  } else if (me.isUnsupportedRepository(record)) {
                    tip.items.get(0).body.update('<span><h2>Repository Health Check Unavailable</h2>A Repository Health Check (RHC) ' +
                      'cannot be performed by the Sonatype CLM service on this repository, because it is an unsupported type or out of service.<br><br>' +
                      '<a href="http://links.sonatype.com/products/clm/rhc/home" ' +
                      'target="_blank">How the Sonatype CLM Repository Health Check can help you make better software faster</a></span>');
                    return;
                  } else if (healthCheckData) {
                    if (healthCheckData.status == 'disabled') {
                      if (Sonatype.HealthCheckUtil.hasEditPerm()) {
                        tip.items.get(0).body.update('<span><h2>Repository Health Check Analysis</h2>Click this button to request a Repository Health Check (RHC) ' +
                          'by the Sonatype CLM service.  The process is non-invasive and non-disruptive.  Sonatype CLM ' +
                          'will return actionable quality, security, and licensing information about the open source components in the repository.' +
                          '<br><br><a href="http://links.sonatype.com/products/clm/rhc/home" ' +
                          'target="_blank">How the Sonatype CLM Repository Health Check can help you make better software faster</a></span>');
                      } else {
                        tip.items.get(0).body.update('<span><h2>Insufficient Permissions to Analyze a Repository</h2>' +
                          'To analyze a repository your user account must have permissions to start analysis.</span>');
                      }
                      return;
                    } else if (healthCheckData.status == 'analyzing') {
                      tip.items.get(0).body.update('<span><h2>The Analysis is Under Way</h2>The contents of your repository are being analyzed.  ' +
                        'This process should only take a few minutes.<br><br>When complete, the ANALYZING button will be ' +
                        'replaced by a set of icons that indicate how many security and licensing issues were discovered.' +
                        '<br><br>Hover your mouse over these icons to see additional information about the issues that were found.</span>');
                      return;
                    }
                  }
                }
                return false;
              }
            }
          });
      };

      Sonatype.Events.addListener('repositoryPanelInit', function(repoPanel) {
          if (Sonatype.HealthCheckUtil.hasReadPerm()) {
              me.reset({
                  grid : repoPanel.gridPanel
              });
          }
          return true;
      });

      Sonatype.Events.addListener('repositoryGridInit', function(repoPanel, columns, toolbar) {
          if (Sonatype.HealthCheckUtil.hasReadPerm()) {
              me.updateColumns(columns);
          }
          return true;
      });

      Sonatype.Events.addListener('healthCheckStatusUpdated', function(repositoryId, enabled, record) {
          if (!record) {
              record = me.findRecord(me.grid.getStore(), repositoryId);
          }
          me.updateGridColumn(record, enabled);
          return true;
      });
  };

  Ext.onReady(function() {
      Sonatype.HealthCheckRepositoryContrib = new HealthCheckRepositoryContrib();
  });
});
