/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.timeline.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Path;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.sonatype.nexus.util.file.DirSupport;
import org.sonatype.timeline.TimelineCallback;
import org.sonatype.timeline.TimelineConfiguration;
import org.sonatype.timeline.TimelineRecord;
import org.sonatype.timeline.proto.TimeLineRecordProtos;

public class DefaultTimelinePersistor {
    private static final String V3_DATA_FILE_NAME_PREFIX = "timeline.";
    private static final String V3_DATA_FILE_NAME_SUFFIX = "-v3.dat";
    private static final String V3_DATA_FILE_NAME_DATE_FORMAT = "yyyy-MM-dd.HH-mm-ssZ";
    private static final Pattern V3_DATA_FILE_NAME_PATTERN = Pattern.compile("^" + "timeline.".replace(".", "\\.") + "(\\d{4}-\\d{2}-\\d{2}\\.\\d{2}-\\d{2}-\\d{2}[+-]\\d{4})" + "-v3.dat".replace(".", "\\.") + "$");
    private int rollingIntervalMillis;
    private File persistDirectory;
    private long lastRolledTimestamp = 0L;
    private File lastRolledFile;

    protected synchronized void setConfiguration(TimelineConfiguration configuration) {
        if (!configuration.getPersistDirectory().exists()) {
            try {
                DirSupport.mkdir((Path)configuration.getPersistDirectory().toPath());
            }
            catch (IOException e) {
                Throwables.propagate((Throwable)e);
            }
        }
        this.persistDirectory = configuration.getPersistDirectory();
        this.rollingIntervalMillis = configuration.getPersistRollingIntervalMillis();
    }

    protected synchronized void persist(TimelineRecord ... records) throws IOException {
        this.verify(records);
        try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(this.getDataFile(), true));){
            for (TimelineRecord record : records) {
                this.toProto(record).writeDelimitedTo(out);
            }
            ((OutputStream)out).flush();
        }
    }

    protected File getDataFile() throws IOException {
        long now = System.currentTimeMillis();
        if (this.lastRolledTimestamp == 0L || now - this.lastRolledTimestamp > (long)(this.rollingIntervalMillis * 1000)) {
            this.lastRolledTimestamp = now;
            this.lastRolledFile = new File(this.persistDirectory, this.buildTimestampedFileName(now));
            this.lastRolledFile.createNewFile();
        }
        return this.lastRolledFile;
    }

    @VisibleForTesting
    protected void readAll(TimelineCallback callback) throws IOException {
        this.readAllSinceDays(Integer.MAX_VALUE, callback);
    }

    protected void readAllSinceDays(int days, TimelineCallback callback) throws IOException {
        List<File> result = this.collectFiles(days, true);
        int filePtr = 0;
        Iterator<TimelineRecord> currentIterator = null;
        while (true) {
            if (currentIterator != null && currentIterator.hasNext()) {
                if (callback.processNext((TimelineRecord)currentIterator.next())) continue;
                break;
            }
            if (filePtr >= result.size()) break;
            File file = result.get(filePtr);
            currentIterator = this.readFile(file);
            ++filePtr;
        }
    }

    protected List<File> collectFiles(int days, boolean youngerThan) {
        File[] files = this.persistDirectory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String fname) {
                return V3_DATA_FILE_NAME_PATTERN.matcher(fname).matches();
            }
        });
        if (files == null || files.length == 0) {
            return Collections.emptyList();
        }
        Arrays.sort(files, new Comparator<File>(){

            @Override
            public int compare(File f1, File f2) {
                long f2ts;
                long f1ts = DefaultTimelinePersistor.this.getTimestampedFileNameTimestamp(f1);
                long result = -(f1ts - (f2ts = DefaultTimelinePersistor.this.getTimestampedFileNameTimestamp(f2)));
                if (result < 0L) {
                    return -1;
                }
                if (result > 0L) {
                    return 1;
                }
                return 0;
            }
        });
        long oldestFileTimestampThreshold = Integer.MAX_VALUE == days ? 0L : this.getTimestampedFileNameTimestamp(files[0]) - (long)days * 24L * 60L * 60L * 1000L;
        ArrayList<File> result = new ArrayList<File>();
        for (File file : files) {
            if (youngerThan) {
                if (oldestFileTimestampThreshold > this.getTimestampedFileNameTimestamp(file)) break;
                result.add(file);
                continue;
            }
            if (oldestFileTimestampThreshold <= this.getTimestampedFileNameTimestamp(file)) continue;
            result.add(file);
        }
        return result;
    }

    protected Iterator<TimelineRecord> readFile(File file) {
        ArrayList<TimelineRecord> result = new ArrayList<TimelineRecord>();
        try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));){
            TimelineRecord rec = this.fromProto(TimeLineRecordProtos.TimeLineRecord.parseDelimitedFrom(in));
            while (rec != null) {
                result.add(rec);
                rec = this.fromProto(TimeLineRecordProtos.TimeLineRecord.parseDelimitedFrom(in));
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return result.iterator();
    }

    protected TimelineRecord fromProto(TimeLineRecordProtos.TimeLineRecord rec) {
        if (rec == null) {
            return null;
        }
        HashMap<String, String> dataMap = new HashMap<String, String>();
        for (TimeLineRecordProtos.TimeLineRecord.Data data : rec.getDataList()) {
            dataMap.put(data.getKey(), data.getValue());
        }
        return new TimelineRecord(rec.getTimestamp(), rec.getType(), rec.getSubType(), dataMap);
    }

    protected int purge(int days) throws IOException {
        List<File> purgedFiles = this.collectFiles(days, false);
        Collections.reverse(purgedFiles);
        for (File file : purgedFiles) {
            if (file.delete()) continue;
        }
        return purgedFiles.size();
    }

    protected String buildTimestampedFileName(long timestamp) {
        SimpleDateFormat dateFormat = new SimpleDateFormat(V3_DATA_FILE_NAME_DATE_FORMAT);
        StringBuilder fileName = new StringBuilder();
        fileName.append(V3_DATA_FILE_NAME_PREFIX).append(dateFormat.format(new Date(timestamp))).append(V3_DATA_FILE_NAME_SUFFIX);
        return fileName.toString();
    }

    protected long getTimestampedFileNameTimestamp(File file) {
        Matcher v3FnMatcher = V3_DATA_FILE_NAME_PATTERN.matcher(file.getName());
        if (v3FnMatcher.matches()) {
            String datePattern = v3FnMatcher.group(1);
            try {
                return new SimpleDateFormat(V3_DATA_FILE_NAME_DATE_FORMAT).parse(datePattern).getTime();
            }
            catch (ParseException parseException) {
                // empty catch block
            }
        }
        return file.lastModified();
    }

    protected TimeLineRecordProtos.TimeLineRecord toProto(TimelineRecord record) {
        TimeLineRecordProtos.TimeLineRecord.Builder builder = TimeLineRecordProtos.TimeLineRecord.newBuilder();
        builder.setTimestamp(record.getTimestamp());
        builder.setType(record.getType());
        builder.setSubType(record.getSubType());
        for (Map.Entry<String, String> entry : record.getData().entrySet()) {
            builder.addData(TimeLineRecordProtos.TimeLineRecord.Data.newBuilder().setKey(entry.getKey()).setValue(entry.getValue()).build());
        }
        return builder.build();
    }

    protected void verify(TimelineRecord ... records) throws IOException {
        if (records.length == 0) {
            throw new IllegalArgumentException("Timeline records array is empty");
        }
        for (TimelineRecord record : records) {
            if (record == null) {
                throw new IllegalArgumentException("Timeline record is null");
            }
            Map<String, String> data = record.getData();
            if (data == null) {
                return;
            }
            for (Map.Entry<String, String> entry : data.entrySet()) {
                if (entry.getKey() == null) {
                    throw new IllegalArgumentException("Timeline record contains invalid data: key is null.");
                }
                if (entry.getValue() != null) continue;
                throw new IllegalArgumentException("Timeline record contains invalid data: value is null.");
            }
        }
    }
}

