/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.util.file;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.Set;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonatype.sisu.goodies.common.SimpleFormat;

public final class DirSupport {
    private static final Logger log = LoggerFactory.getLogger(DirSupport.class);
    public static final Set<FileVisitOption> DEFAULT_FILE_VISIT_OPTIONS = EnumSet.of(FileVisitOption.FOLLOW_LINKS);

    private DirSupport() {
    }

    public static void mkdir(Path dir) throws IOException {
        block2: {
            try {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            catch (FileAlreadyExistsException e) {
                if (Files.isDirectory(dir, new LinkOption[0])) break block2;
                throw e;
            }
        }
    }

    public static void mkdir(File dir) throws IOException {
        DirSupport.mkdir(dir.toPath());
    }

    public static void clean(Path dir) throws IOException {
        DirSupport.validateDirectory(dir);
        Files.walkFileTree(dir, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static boolean cleanIfExists(Path dir) throws IOException {
        Preconditions.checkNotNull((Object)dir);
        if (Files.exists(dir, new LinkOption[0])) {
            DirSupport.clean(dir);
            return true;
        }
        return false;
    }

    public static void empty(final Path dir) throws IOException {
        DirSupport.validateDirectory(dir);
        Files.walkFileTree(dir, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path f, BasicFileAttributes attrs) throws IOException {
                Files.delete(f);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path d, IOException exc) throws IOException {
                if (exc != null) {
                    throw exc;
                }
                if (dir != d) {
                    Files.delete(d);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static boolean emptyIfExists(Path dir) throws IOException {
        Preconditions.checkNotNull((Object)dir);
        if (Files.exists(dir, new LinkOption[0])) {
            DirSupport.empty(dir);
            return true;
        }
        return false;
    }

    public static void delete(Path dir) throws IOException {
        DirSupport.delete(dir, null);
    }

    public static void delete(Path dir, final @Nullable Predicate<Path> excludeFilter) throws IOException {
        DirSupport.validateDirectoryOrFile(dir);
        if (Files.isDirectory(dir, new LinkOption[0])) {
            Files.walkFileTree(dir, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (excludeFilter != null && excludeFilter.apply((Object)dir)) {
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (exc != null) {
                        throw exc;
                    }
                    boolean needsDelete = true;
                    if (excludeFilter != null) {
                        try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(dir);){
                            if (dirStream.iterator().hasNext()) {
                                needsDelete = false;
                            }
                        }
                    }
                    if (needsDelete) {
                        Files.delete(dir);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        } else {
            Files.delete(dir);
        }
    }

    public static boolean deleteIfExists(Path dir) throws IOException {
        return DirSupport.deleteIfExists(dir, null);
    }

    public static boolean deleteIfExists(Path dir, @Nullable Predicate<Path> excludeFilter) throws IOException {
        Preconditions.checkNotNull((Object)dir);
        if (Files.exists(dir, new LinkOption[0])) {
            DirSupport.delete(dir, excludeFilter);
            return true;
        }
        return false;
    }

    public static void copy(Path from, Path to) throws IOException {
        DirSupport.copy(from, to, null);
    }

    public static void copy(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        DirSupport.validateDirectoryOrFile(from);
        Preconditions.checkNotNull((Object)to);
        if (Files.isDirectory(from, new LinkOption[0])) {
            Files.walkFileTree(from, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, new CopyVisitor(from, to, excludeFilter));
        } else {
            DirSupport.mkdir(to.getParent());
            Files.copy(from, to, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static boolean copyIfExists(Path from, Path to) throws IOException {
        return DirSupport.copyIfExists(from, to, null);
    }

    public static boolean copyIfExists(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        Preconditions.checkNotNull((Object)from);
        if (Files.exists(from, new LinkOption[0])) {
            DirSupport.copy(from, to, excludeFilter);
            return true;
        }
        return false;
    }

    private static void sameFileStoreMove(Path from, Path to) throws IOException {
        Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
    }

    private static void crossFileStoreMove(Path from, Path to) throws IOException {
        DirSupport.copyDeleteMove(from, to, null);
    }

    private static boolean areOnSameFileStore(Path from, Path to) {
        try {
            Path toExistingParent;
            FileStore fromStore = Files.getFileStore(from);
            for (toExistingParent = to.normalize(); toExistingParent != null && !Files.exists(toExistingParent, new LinkOption[0]); toExistingParent = toExistingParent.getParent()) {
            }
            if (toExistingParent != null) {
                FileStore toStore = Files.getFileStore(toExistingParent);
                return fromStore.equals(toStore);
            }
            log.warn("No ultimate parent path found for '{}'", (Object)to, (Object)new RuntimeException("marker"));
            return false;
        }
        catch (IOException e) {
            return false;
        }
    }

    public static void move(Path from, Path to) throws IOException {
        if (DirSupport.areOnSameFileStore(from, to)) {
            DirSupport.sameFileStoreMove(from, to);
        } else {
            DirSupport.crossFileStoreMove(from, to);
        }
    }

    public static boolean moveIfExists(Path from, Path to) throws IOException {
        Preconditions.checkNotNull((Object)from);
        if (Files.exists(from, new LinkOption[0])) {
            DirSupport.move(from, to);
            return true;
        }
        return false;
    }

    public static void copyDeleteMove(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        DirSupport.copy(from, to, excludeFilter);
        DirSupport.delete(from, excludeFilter);
    }

    public static boolean copyDeleteMoveIfExists(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        Preconditions.checkNotNull((Object)from);
        if (Files.exists(from, new LinkOption[0])) {
            DirSupport.copyDeleteMove(from, to, excludeFilter);
            return true;
        }
        return false;
    }

    public static void apply(Path from, Function<Path, FileVisitResult> func) throws IOException {
        DirSupport.validateDirectory(from);
        Files.walkFileTree(from, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, new FunctionVisitor(func));
    }

    public static void applyToFiles(Path from, Function<Path, FileVisitResult> func) throws IOException {
        DirSupport.validateDirectory(from);
        Files.walkFileTree(from, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, new FunctionFileVisitor(func));
    }

    private static void validateDirectory(Path ... paths) {
        for (Path path : paths) {
            Preconditions.checkNotNull((Object)path, (Object)"Path must be non-null");
            Preconditions.checkArgument((boolean)Files.isDirectory(path, new LinkOption[0]), (Object)SimpleFormat.template((String)"%s is not a directory", (Object[])new Object[]{path}));
        }
    }

    private static void validateDirectoryOrFile(Path ... paths) {
        for (Path path : paths) {
            Preconditions.checkNotNull((Object)path, (Object)"Path must be non-null");
            Preconditions.checkArgument((boolean)Files.exists(path, new LinkOption[0]), (Object)SimpleFormat.template((String)"%s does not exists", (Object[])new Object[]{path}));
        }
    }

    public static class CopyVisitor
    extends SimpleFileVisitor<Path> {
        private final Path from;
        private final Path to;
        private final Predicate<Path> excludeFilter;

        public CopyVisitor(Path from, Path to, @Nullable Predicate<Path> excludeFilter) {
            this.from = from;
            this.to = to;
            this.excludeFilter = excludeFilter;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes a) throws IOException {
            if (this.excludeFilter != null && this.excludeFilter.apply((Object)dir)) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            Path targetPath = this.to.resolve(this.from.relativize(dir));
            if (!Files.exists(targetPath, new LinkOption[0])) {
                DirSupport.mkdir(targetPath);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes a) throws IOException {
            Files.copy(file, this.to.resolve(this.from.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
            return FileVisitResult.CONTINUE;
        }
    }

    public static class FunctionFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Function<Path, FileVisitResult> func;

        public FunctionFileVisitor(Function<Path, FileVisitResult> func) {
            this.func = (Function)Preconditions.checkNotNull(func);
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes a) throws IOException {
            return (FileVisitResult)((Object)this.func.apply((Object)file));
        }
    }

    public static class FunctionVisitor
    extends SimpleFileVisitor<Path> {
        private final Function<Path, FileVisitResult> func;

        public FunctionVisitor(Function<Path, FileVisitResult> func) {
            this.func = (Function)Preconditions.checkNotNull(func);
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes a) throws IOException {
            return (FileVisitResult)((Object)this.func.apply((Object)file));
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
            if (e != null) {
                throw e;
            }
            return (FileVisitResult)((Object)this.func.apply((Object)dir));
        }
    }
}

