/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.configuration;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.StandardCopyOption;
import java.util.HashMap;
import java.util.Objects;
import org.slf4j.Logger;
import org.sonatype.nexus.util.file.DirSupport;
import org.sonatype.sisu.goodies.common.Loggers;
import org.sonatype.sisu.goodies.common.io.FileReplacer;

public class ModelUtils {
    private static final Logger log = Loggers.getLogger(ModelUtils.class);
    public static final Charset DEFAULT_CHARSET = Charsets.UTF_8;

    private ModelUtils() {
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static <E> E load(String currentModelVersion, File file, ModelReader<E> reader, ModelUpgrader ... upgraders) throws CorruptModelException, IOException {
        Preconditions.checkNotNull((Object)currentModelVersion, (Object)"currentModelVersion");
        Preconditions.checkNotNull((Object)file, (Object)"file");
        Preconditions.checkNotNull(reader, (Object)"reader");
        Preconditions.checkNotNull((Object)upgraders, (Object)"upgraders");
        log.info("Loading model {}", (Object)file.getAbsoluteFile(), (Object)currentModelVersion);
        try {
            Object currentFileVersion;
            if (reader instanceof Versioned) {
                String originalFileVersion;
                try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file.toPath(), new OpenOption[0]));){
                    originalFileVersion = ((Versioned)((Object)reader)).readVersion(input);
                }
                if (Strings.isNullOrEmpty((String)originalFileVersion)) {
                    throw new MissingModelVersionException("Passed in model has null version");
                }
                if (!Objects.equals(currentModelVersion, originalFileVersion)) {
                    log.info("Upgrading model {} from version {} to {}", new Object[]{file.getAbsoluteFile(), originalFileVersion, currentModelVersion});
                    currentFileVersion = originalFileVersion;
                    HashMap upgradersMap = Maps.newHashMapWithExpectedSize((int)upgraders.length);
                    for (ModelUpgrader modelUpgrader : upgraders) {
                        upgradersMap.put(modelUpgrader.fromVersion(), modelUpgrader);
                    }
                    FileReplacer fileReplacer = new FileReplacer(file);
                    fileReplacer.setDeleteBackupFile(true);
                    ModelUpgrader upgrader = (ModelUpgrader)upgradersMap.get(currentFileVersion);
                    Files.copy(file.toPath(), new File(file.getParentFile(), file.getName() + ".old").toPath(), StandardCopyOption.REPLACE_EXISTING);
                    while (upgrader != null && !Objects.equals(currentModelVersion, currentFileVersion)) {
                        try {
                            fileReplacer.replace((FileReplacer.ContentWriter)new ModelUpgraderAdapter(file, upgrader));
                        }
                        catch (CorruptModelException e) {
                            CorruptModelException corruptModelException = new CorruptModelException(String.format("Model %s detected as corrupt during upgrade from version %s to version %s", file.getAbsolutePath(), upgrader.fromVersion(), upgrader.toVersion()), e);
                            throw corruptModelException;
                        }
                        catch (IOException e) {
                            IOException iOException = new IOException(String.format("IO problem during upgrade from version %s to version %s of %s", upgrader.fromVersion(), upgrader.toVersion(), file.getAbsolutePath()), e);
                            throw iOException;
                        }
                        currentFileVersion = upgrader.toVersion();
                        upgrader = (ModelUpgrader)upgradersMap.get(currentFileVersion);
                    }
                    if (!Objects.equals(currentModelVersion, currentFileVersion)) {
                        throw new IOException(String.format("Could not upgrade model %s to version %s, is upgraded to %s, originally was %s, available upgraders exists for versions %s", file.getAbsolutePath(), currentModelVersion, currentFileVersion, originalFileVersion, upgradersMap.keySet()));
                    }
                }
            }
            BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file.toPath(), new OpenOption[0]));
            currentFileVersion = null;
            try {
                E model;
                E e = model = reader.read(input);
                return e;
            }
            catch (Throwable throwable) {
                currentFileVersion = throwable;
                throw throwable;
            }
            finally {
                if (input != null) {
                    if (currentFileVersion != null) {
                        try {
                            ((InputStream)input).close();
                        }
                        catch (Throwable throwable) {
                            ((Throwable)currentFileVersion).addSuppressed(throwable);
                        }
                    } else {
                        ((InputStream)input).close();
                    }
                }
            }
        }
        catch (NoSuchFileException e) {
            FileNotFoundException fnf = new FileNotFoundException(e.getMessage());
            fnf.initCause(e);
            throw fnf;
        }
    }

    public static <E> void save(final E model, File file, final ModelWriter<E> writer) throws IOException {
        Preconditions.checkNotNull(model, (Object)"model");
        Preconditions.checkNotNull((Object)file, (Object)"File");
        Preconditions.checkNotNull(writer, (Object)"ModelWriter");
        log.info("Saving model {}", (Object)file.getAbsoluteFile());
        DirSupport.mkdir(file.getParentFile().toPath());
        File backupFile = new File(file.getParentFile(), file.getName() + ".bak");
        FileReplacer fileReplacer = new FileReplacer(file);
        fileReplacer.setDeleteBackupFile(false);
        fileReplacer.replace(new FileReplacer.ContentWriter(){

            public void write(BufferedOutputStream output) throws IOException {
                writer.write(output, model);
                output.flush();
            }
        });
        if (Files.exists(fileReplacer.getBackupFile().toPath(), new LinkOption[0])) {
            Files.copy(fileReplacer.getBackupFile().toPath(), backupFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            Files.delete(fileReplacer.getBackupFile().toPath());
        }
    }

    private static class ModelUpgraderAdapter
    implements FileReplacer.ContentWriter {
        private final File file;
        private final ModelUpgrader modelUpgrader;

        private ModelUpgraderAdapter(File file, ModelUpgrader modelUpgrader) {
            this.file = (File)Preconditions.checkNotNull((Object)file);
            this.modelUpgrader = (ModelUpgrader)Preconditions.checkNotNull((Object)modelUpgrader);
        }

        public void write(BufferedOutputStream output) throws IOException {
            try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(this.file.toPath(), new OpenOption[0]));){
                this.modelUpgrader.upgrade(input, output);
            }
        }
    }

    public static class MissingModelVersionException
    extends CorruptModelException {
        public MissingModelVersionException(String message) {
            super(message);
        }
    }

    public static class CorruptModelException
    extends RuntimeException {
        public CorruptModelException(String message) {
            super(message);
        }

        public CorruptModelException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static abstract class CharacterModelUpgrader
    extends ModelUpgrader {
        private final Charset charset;

        protected CharacterModelUpgrader(String fromVersion, String toVersion) {
            this(fromVersion, toVersion, DEFAULT_CHARSET);
        }

        protected CharacterModelUpgrader(String fromVersion, String toVersion, Charset charset) {
            super(fromVersion, toVersion);
            this.charset = (Charset)Preconditions.checkNotNull((Object)charset);
        }

        @Override
        public void upgrade(InputStream input, OutputStream output) throws IOException, CorruptModelException {
            this.upgrade(new InputStreamReader(input, this.charset), new OutputStreamWriter(output, this.charset));
        }

        public abstract void upgrade(Reader var1, Writer var2) throws IOException, CorruptModelException;
    }

    public static abstract class CharacterModelWriter<E>
    implements ModelWriter<E> {
        private final Charset charset;

        protected CharacterModelWriter() {
            this(DEFAULT_CHARSET);
        }

        protected CharacterModelWriter(Charset charset) {
            this.charset = (Charset)Preconditions.checkNotNull((Object)charset);
        }

        @Override
        public void write(OutputStream output, E model) throws IOException {
            this.write(new OutputStreamWriter(output, this.charset), model);
        }

        public abstract void write(Writer var1, E var2) throws IOException;
    }

    public static abstract class CharacterModelReader<E>
    implements ModelReader<E> {
        protected final Charset charset;

        protected CharacterModelReader() {
            this(DEFAULT_CHARSET);
        }

        protected CharacterModelReader(Charset charset) {
            this.charset = (Charset)Preconditions.checkNotNull((Object)charset);
        }

        @Override
        public E read(InputStream input) throws IOException, CorruptModelException {
            return this.read(new InputStreamReader(input, this.charset));
        }

        public abstract E read(Reader var1) throws IOException, CorruptModelException;
    }

    public static abstract class ModelUpgrader {
        private final String fromVersion;
        private final String toVersion;

        public ModelUpgrader(String fromVersion, String toVersion) {
            this.fromVersion = (String)Preconditions.checkNotNull((Object)fromVersion);
            this.toVersion = (String)Preconditions.checkNotNull((Object)toVersion);
        }

        public final String fromVersion() {
            return this.fromVersion;
        }

        public final String toVersion() {
            return this.toVersion;
        }

        public abstract void upgrade(InputStream var1, OutputStream var2) throws IOException, CorruptModelException;
    }

    public static interface ModelWriter<E> {
        public void write(OutputStream var1, E var2) throws IOException;
    }

    public static interface Versioned {
        public String readVersion(InputStream var1) throws IOException, CorruptModelException;
    }

    public static interface ModelReader<E> {
        public E read(InputStream var1) throws IOException, CorruptModelException;
    }
}

